const prisma = require("../config/db");

async function getAllPosts() {
  return prisma.post.findMany({
    include: {
      category: true,
      author: true,
      tags: true,
      seo: true
    },
    orderBy: { publishedAt: "desc" }
  });
}

async function getPostBySlug(slug) {
  return prisma.post.findUnique({
    where: { slug },
    include: {
      category: true,
      author: true,
      tags: true,
      seo: true
    }
  });
}

async function createPost(data) {
  return prisma.post.create({
    data: {
      title: data.title,
      slug: data.slug,
      excerpt: data.excerpt,
      content: data.content,
      image: data.image || null,
      categoryId: data.categoryId,
      authorId: data.authorId,
      readTime: data.readTime || null,
      publishedAt: data.publishedAt ? new Date(data.publishedAt) : null,

      // 🔥 new flags
      isHero:        data.isHero || false,
      isFeatured:    data.isFeatured || false,
      isMustRead:    data.isMustRead || false,
      isTrending:    data.isTrending || false,
      isTopThisWeek: data.isTopThisWeek || false,

      tags: {
        connect: data.tags.map((id) => ({ id }))
      }
    }
  });
}


async function updatePost(id, data) {
  return prisma.post.update({
    where: { id: Number(id) },
    data: {
      title: data.title,
      slug: data.slug,
      excerpt: data.excerpt,
      content: data.content,
      image: data.image || null,
      categoryId: data.categoryId,
      authorId: data.authorId,
      readTime: data.readTime || null,
      publishedAt: data.publishedAt ? new Date(data.publishedAt) : null,

      isHero:        data.isHero,
      isFeatured:    data.isFeatured,
      isMustRead:    data.isMustRead,
      isTrending:    data.isTrending,
      isTopThisWeek: data.isTopThisWeek,

      tags: data.tags
        ? {
            set: [],
            connect: data.tags.map((id) => ({ id }))
          }
        : undefined
    }
  });
}


async function deletePost(id) {
  return prisma.post.delete({
    where: { id: Number(id) }
  });
}

async function searchPosts(query) {
  return prisma.post.findMany({
    where: {
      OR: [
        { title: { contains: query, mode: "insensitive" } },
        { excerpt: { contains: query, mode: "insensitive" } },
        { content: { contains: query, mode: "insensitive" } }
      ]
    },
    include: {
      category: true,
      author: true,
      tags: true,
      seo: true
    },
    orderBy: { publishedAt: "desc" }
  });
}



function buildPostFilters(query) {
  const {
    categoryId,
    categorySlug,
    tagId,
    tagSlug,
    authorId,
    section,
    search
  } = query;

  const where = {};

  if (categoryId) {
    where.categoryId = Number(categoryId);
  }

  if (authorId) {
    where.authorId = Number(authorId);
  }

  if (categorySlug) {
    where.category = { slug: categorySlug };
  }

  if (tagId) {
    where.tags = { some: { id: Number(tagId) } };
  }

  if (tagSlug) {
    where.tags = { some: { slug: tagSlug } };
  }

  if (search) {
    where.OR = [
      { title: { contains: search, mode: "insensitive" } },
      { excerpt: { contains: search, mode: "insensitive" } },
      { content: { contains: search, mode: "insensitive" } }
    ];
  }

  // section = hero | featured | must_read | trending | top_this_week
  if (section) {
    switch (section) {
      case "hero":
        where.isHero = true;
        break;
      case "featured":
        where.isFeatured = true;
        break;
      case "must_read":
        where.isMustRead = true;
        break;
      case "trending":
        where.isTrending = true;
        break;
      case "top_this_week":
        where.isTopThisWeek = true;
        break;
    }
  }

  // Only published posts by default
  where.publishedAt = { not: null };

  return where;
}


async function getPostsAdvanced(query) {
  const page = parseInt(query.page, 10) || 1;
  const limit = parseInt(query.limit, 10) || 10;
  const skip = (page - 1) * limit;

  const sort = query.sort || "latest";
  let orderBy;

  // sort = latest | oldest | views | random(FYI: Prisma doesn't random easily across all dbs, so we skip that here)
  switch (sort) {
    case "oldest":
      orderBy = { publishedAt: "asc" };
      break;
    case "views":
      orderBy = { viewCount: "desc" };
      break;
    default:
      orderBy = { publishedAt: "desc" };
  }

  const where = buildPostFilters(query);

  const posts = await prisma.post.findMany({
    where,
    include: {
      category: true,
      author: true,
      tags: true,
      seo: true
    },
    skip,
    take: limit,
    orderBy
  });

  const total = await prisma.post.count({ where });

  return {
    posts,
    page,
    limit,
    total,
    totalPages: Math.ceil(total / limit)
  };
}


module.exports = {
  getAllPosts,
  getPostBySlug,
  createPost,
  updatePost,
  deletePost,
  searchPosts,
  getPostsAdvanced    
};
