// src/controllers/admin/auth.controller.js
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const config = require("../../config/env");
const adminService = require("../../services/admin.service");
const { validateLogin } = require("../../validators/login.validator");
const { success, error } = require("../../utils/response");

// LOGIN
exports.login = async (req, res, next) => {
  try {
    const { valid, errors } = validateLogin(req.body);
    if (!valid) return error(res, errors.join(", "), 400);

    const admin = await adminService.findAdminByEmail(req.body.email);
    if (!admin) return error(res, "Invalid credentials", 401);

    const match = await bcrypt.compare(req.body.password, admin.password);
    if (!match) return error(res, "Invalid credentials", 401);

    const token = jwt.sign(
      { id: admin.id, email: admin.email, role: admin.role },
      config.JWT_SECRET,          // <— uses JWT_SECRET (UPPERCASE)
      { expiresIn: "1d" }
    );

    return success(res, { token }, "Login successful");
  } catch (err) {
    next(err);
  }
};

// CREATE ADMIN (setup route)
exports.createAdmin = async (req, res, next) => {
  try {
    const { email, password, role } = req.body;

    if (!email || !password) {
      return error(res, "Email and password are required", 400);
    }

    // Allow creating admin only if none exists yet
    const count = await adminService.getAdminCount();
    if (count > 0) {
      return error(
        res,
        "Admin already exists. This endpoint is only for initial setup.",
        403
      );
    }

    const existing = await adminService.findAdminByEmail(email);
    if (existing) {
      return error(res, "Admin with this email already exists", 400);
    }

    const hashedPassword = await bcrypt.hash(password, 10);

    const admin = await adminService.createAdmin({
      email,
      password: hashedPassword,
      role: role || "admin"
    });

    return success(
      res,
      { id: admin.id, email: admin.email, role: admin.role },
      "Admin created successfully",
      201
    );
  } catch (err) {
    next(err);
  }
};
